
//--------------------------------------------------------------------//
//--- Dullard: trace all memory data accesses.           dl_main.c ---//
//--------------------------------------------------------------------//

/*
   This file is part of Memtrace, a Valgrind tool for tracing memory
   data accesses.

   Copyright (C) 2004 Nicholas Nethercote
      njn25@cam.ac.uk

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA
   02111-1307, USA.

   The GNU General Public License is contained in the file COPYING.
*/

/*
   Dullard is a good starting point for building Valgrind tools that need to
   act on memory loads and stores.  It also could be used as is, with its
   output used as input to a post-mortem processing step.  However, because
   memory traces can be very large, online analysis is generally better.

   Dullard prints memory data access traces that look like this:

      read     : size=4, addr=0xAFEFE3F8
      write    : size=4, addr=0xAFEFE3EC
      modify   : size=4, addr=0xAFEFDED8
      readwrite: size=4, addr1=0x3A9D3630, addr2=0xAFEFE3E8

   "read" means the instruction reads a memory location
      eg. movl (%eax), %ebx      -- reads (%eax)

   "write" means the instruction writes a memory location
      eg. movl %eax, (%ebx)      -- writes (%ebx)

   "modify" means the instruction reads and writes (modifies) a memory location
      eg. incl (%ecx)            -- modifies (%ecx)

   "readwrite" means the instruction reads one address, and writes another
      eg. call*l (%edx)          -- reads (%edx), writes -4(%esp)
          pushl (%edx)           -- reads (%edx), writes -4(%esp)
          movsw                  -- reads (%esi), writes (%edi)

   Dullard gives good traces, but they are not perfect, for the following
   reasons:

   - It does not trace into the OS kernel, so system calls and other kernel
     operations (eg. some scheduling and signal handling code) are ignored.

   - Valgrind replaces some code, notably parts of code for thread and
     scheduling operations, and signal handling.  This code is not traced.

   - There is no consideration of virtual-to-physical address mapping.
     This may not matter for many purposes.

   - Valgrind modifies the instruction stream in some very minor ways.  For
     example, the bts, btc, btr instructions are incorrectly considered to
     always touch memory (this is a consequence of these instructions being
     very difficult to simulate).

   - Valgrind tools layout memory differently to normal programs, so the
     addresses you get will not be typical.  For example, on x86/Linux the
     stack usually begins at 0xbfffffff and grows down.  With Dullard, it
     will be in a different place (at the time of writing, on my machine, it
     begins at 0xafefffff, but this may change in the future if Valgrind's
     memory layout changes).  Thus Dullard is suitable for getting relative
     memory traces -- eg. if you want to analyse locality of memory accesses
     -- but is not good if absolute addresses are important.

   Despite all these warnings, Dullard's results should be good enough for a
   wide range of purposes.  For example, Cachegrind shares all the above
   shortcomings and it is still useful.
*/

#include "vg_skin.h"
//#include "vg_profile.c"

//------------------------------------------------------------//
///-- Memory tracing functions                             ---//
///-----------------------------------------------------------//

static __attribute__ ((regparm (2)))
void mem_read(UInt data_size, Addr data_addr)
{
   VG_(printf)("read     : size=%u, addr=%p\n", data_size, data_addr);
}

static __attribute__ ((regparm (2)))
void mem_write(UInt data_size, Addr data_addr)
{
   VG_(printf)("write    : size=%u, addr=%p\n", data_size, data_addr);
}

static __attribute__ ((regparm (2)))
void mem_modify(UInt data_size, Addr data_addr)
{
   VG_(printf)("modify   : size=%u, addr=%p\n", data_size, data_addr);
}

static __attribute__ ((regparm (3)))
void mem_readwrite(UInt data_size, Addr data_addr1, Addr data_addr2)
{
   VG_(printf)("readwrite: size=%u, addr1=%p, addr2=%p\n", 
               data_size, data_addr1, data_addr2);
}

//------------------------------------------------------------//
//--- Instrumentation                                      ---//
//------------------------------------------------------------//

Bool is_valid_data_size(Int data_size)
{
   return (data_size ==  1 || data_size ==   2 || data_size ==   4
        || data_size ==  8 || data_size ==  10 || data_size ==  16 
        || data_size == 28 || data_size == 108 || data_size == 512);
}

// Instrumentation for the end of each x86 instruction.
void end_of_x86_instr(UCodeBlock* cb, UInt data_size,
                      Int t_read,  Int t_read_addr, 
                      Int t_write, Int t_write_addr)
{
#define IS_(X)      (INVALID_TEMPREG != t_##X##_addr)
#define INV(qqt)    (INVALID_TEMPREG == (qqt))

   // Decide what kind of x86 instruction it is, and call the right function
   if (!IS_(read) && !IS_(write)) {
      sk_assert( 0 == data_size );
      sk_assert(INV(t_read) && INV(t_write));
      // no instrumentation

   } else if (IS_(read) && !IS_(write)) {
      sk_assert( is_valid_data_size(data_size) );
      sk_assert(!INV(t_read) && INV(t_write));
      VG_(ccall_LR_0)(cb, (Addr)mem_read, data_size, t_read_addr, 2);

   } else if (!IS_(read) && IS_(write)) {
      sk_assert( is_valid_data_size(data_size) );
      sk_assert(INV(t_read) && !INV(t_write));
      VG_(ccall_LR_0)(cb, (Addr)mem_write, data_size, t_write_addr, 2);

   } else {
      sk_assert(IS_(read) && IS_(write));
      sk_assert( is_valid_data_size(data_size) );
      sk_assert(!INV(t_read) && !INV(t_write));
      if (t_read == t_write) {
         VG_(ccall_LR_0)(cb, (Addr)mem_modify, data_size, t_read_addr, 2);
      } else {
         VG_(ccall_LRR_0)(cb, (Addr)mem_readwrite, data_size, 
                          t_read_addr, t_write_addr, 3);
      }
   }
#undef IS_
#undef INV
}

UCodeBlock* SK_(instrument)(UCodeBlock* cb_in, Addr orig_addr)
{
   UCodeBlock* cb;
   UInstr*     u_in;
   Int         i;
   Int         t_read_addr, t_write_addr, t_read, t_write;
   UInt        data_size = 0;
   Bool        instrumented_Jcc = False;

   t_read_addr = t_write_addr = t_read = t_write = INVALID_TEMPREG;

   cb = VG_(setup_UCodeBlock)(cb_in);

   for (i = 0; i < VG_(get_num_instrs)(cb_in); i++) {
      u_in = VG_(get_instr)(cb_in, i);

      // We want to instrument each x86 instruction with a call to the
      // appropriate tracing function, which depends on whether the
      // instruction does memory data reads/writes.  x86 instructions can
      // end in three ways, and this is how they are instrumented:
      //
      // 1. UCode, INCEIP   --> UCode, Instrumentation, INCEIP
      // 2. UCode, JMP      --> UCode, Instrumentation, JMP
      // 3. UCode, Jcc, JMP --> UCode, Instrumentation, Jcc, JMP
      //
      // The last UInstr in a BB is always a JMP.  Jccs, when they appear,
      // are always second last.  This is checked with assertions.
      // Instrumentation must go before any jumps.  (JIFZ is the exception;
      // if a JIFZ succeeds, no simulation is done for the instruction.)
      //
      // x86 instruction sizes are obtained from INCEIPs (for case 1) or
      // from .extra4b field of the final JMP (for case 2 & 3).

      if (instrumented_Jcc) sk_assert(u_in->opcode == JMP);

      switch (u_in->opcode) {

         // For memory-ref instrs, copy the data_addr into a temporary to be
         // passed to the mem_* helper at the end of the instruction.
         case LOAD: 
         case SSE3ag_MemRd_RegWr:
            t_read      = u_in->val1;
            t_read_addr = newTemp(cb);
            uInstr2(cb, MOV, 4, TempReg, u_in->val1,  TempReg, t_read_addr);
            data_size = u_in->size;
            VG_(copy_UInstr)(cb, u_in);
            break;

         case FPU_R:
         case MMX2_MemRd:
            t_read      = u_in->val2;
            t_read_addr = newTemp(cb);
            uInstr2(cb, MOV, 4, TempReg, u_in->val2,  TempReg, t_read_addr);
            data_size = u_in->size; 
            VG_(copy_UInstr)(cb, u_in);
            break;
            break;

         case MMX2a1_MemRd:
         case SSE2a_MemRd:
         case SSE2a1_MemRd:
         case SSE3a_MemRd:
         case SSE3a1_MemRd:
            t_read = u_in->val3;
            t_read_addr = newTemp(cb);
            uInstr2(cb, MOV, 4, TempReg, u_in->val3,  TempReg, t_read_addr);
            data_size = u_in->size;
            VG_(copy_UInstr)(cb, u_in);
            break;

         // Note that we must set t_write_addr even for mod instructions;
         // That's how the code above determines whether it does a write.
         // Without it, it would think a mod instruction is a read.
         // As for the MOV, if it's a mod instruction it's redundant, but it's
         // not expensive and mod instructions are rare anyway.
         case STORE:
         case FPU_W:
         case MMX2_MemWr:
            t_write      = u_in->val2;
            t_write_addr = newTemp(cb);
            uInstr2(cb, MOV, 4, TempReg, u_in->val2, TempReg, t_write_addr);
            data_size = u_in->size;
            VG_(copy_UInstr)(cb, u_in);
            break;

         case SSE2a_MemWr:
         case SSE3a_MemWr:
            t_write = u_in->val3;
            t_write_addr = newTemp(cb);
            uInstr2(cb, MOV, 4, TempReg, u_in->val3, TempReg, t_write_addr);
            data_size = u_in->size; 
            VG_(copy_UInstr)(cb, u_in);
            break;

         // JMP, INCEIP: insert instrumentation if the first JMP
         case JMP:
            if (instrumented_Jcc) {
               sk_assert(CondAlways == u_in->cond);
               sk_assert(i+1 == VG_(get_num_instrs)(cb_in));
               VG_(copy_UInstr)(cb, u_in);
               instrumented_Jcc = False;     // rest
               break;
            } else {
               // The first JMP... instrument.
               if (CondAlways != u_in->cond) {
                  sk_assert(i+2 == VG_(get_num_instrs)(cb_in));
                  instrumented_Jcc = True;
               } else {
                  sk_assert(i+1 == VG_(get_num_instrs)(cb_in));
               }
            }
            // Fall-through
         case INCEIP:
            end_of_x86_instr(cb, data_size,
                             t_read, t_read_addr, t_write, t_write_addr);

            // Copy original UInstr (INCEIP or JMP)
            VG_(copy_UInstr)(cb, u_in);

            // Update loop state for next x86 instr
            t_read_addr = t_write_addr = t_read = t_write = INVALID_TEMPREG;
            data_size = 0;
            break;

         default:
            VG_(copy_UInstr)(cb, u_in);
            break;
      }
   }

   VG_(free_UCodeBlock)(cb_in);
   return cb;

#undef INVALID_DATA_SIZE
}

//--------------------------------------------------------------------//
//--- init/fini                                                    ---//
//--------------------------------------------------------------------//

void SK_(pre_clo_init)(void)
{
   VG_(details_name)            ("Dullard");
   VG_(details_version)         (NULL);
   VG_(details_description)     ("a memory data access tracer");
   VG_(details_copyright_author)(
      "Copyright (C) 2004, and GNU GPL'd, by Nicholas Nethercote.");
   VG_(details_bug_reports_to)  (VG_BUGS_TO);
   VG_(details_avg_translation_sizeB) ( 155 );

   VG_(register_compact_helper)((Addr) mem_read);
   VG_(register_compact_helper)((Addr) mem_write);
   VG_(register_compact_helper)((Addr) mem_modify);
   VG_(register_compact_helper)((Addr) mem_readwrite);
}

void SK_(post_clo_init)(void)
{
}

void SK_(fini)(Int exitcode)
{
}

VG_DETERMINE_INTERFACE_VERSION(SK_(pre_clo_init), 0)

//--------------------------------------------------------------------//
//--- end                                                dl_main.c ---//
//--------------------------------------------------------------------//
