#include "bridge.hh"
#include <algorithm>
#include <array>
#include <cassert>
#include <cstddef>
#include <cstdint>
#include <initializer_list>
#include <iterator>
#include <memory>
#include <new>
#include <stdexcept>
#include <string>
#include <type_traits>
#include <utility>
#if __cplusplus >= 201703L
#include <string_view>
#endif
#if __cplusplus >= 202002L
#include <ranges>
#endif

#ifdef __GNUC__
#pragma GCC diagnostic ignored "-Wmissing-declarations"
#ifdef __clang__
#pragma clang diagnostic ignored "-Wdollar-in-identifier-extension"
#endif // __clang__
#endif // __GNUC__

namespace rust {
inline namespace cxxbridge1 {
// #include "rust/cxx.h"

#ifndef CXXBRIDGE1_PANIC
#define CXXBRIDGE1_PANIC
template <typename Exception>
void panic [[noreturn]] (const char *msg);
#endif // CXXBRIDGE1_PANIC

struct unsafe_bitcopy_t;

namespace {
template <typename T>
class impl;
} // namespace

class Opaque;

template <typename T>
::std::size_t size_of();
template <typename T>
::std::size_t align_of();

#ifndef CXXBRIDGE1_RUST_STRING
#define CXXBRIDGE1_RUST_STRING
class String final {
public:
  String() noexcept;
  String(const String &) noexcept;
  String(String &&) noexcept;
  ~String() noexcept;

  String(const std::string &);
  String(const char *);
  String(const char *, std::size_t);
  String(const char16_t *);
  String(const char16_t *, std::size_t);
#ifdef __cpp_char8_t
  String(const char8_t *s);
  String(const char8_t *s, std::size_t len);
#endif

  static String lossy(const std::string &) noexcept;
  static String lossy(const char *) noexcept;
  static String lossy(const char *, std::size_t) noexcept;
  static String lossy(const char16_t *) noexcept;
  static String lossy(const char16_t *, std::size_t) noexcept;

  String &operator=(const String &) & noexcept;
  String &operator=(String &&) & noexcept;

  explicit operator std::string() const;

  const char *data() const noexcept;
  std::size_t size() const noexcept;
  std::size_t length() const noexcept;
  bool empty() const noexcept;

  const char *c_str() noexcept;

  std::size_t capacity() const noexcept;
  void reserve(size_t new_cap) noexcept;

  using iterator = char *;
  iterator begin() noexcept;
  iterator end() noexcept;

  using const_iterator = const char *;
  const_iterator begin() const noexcept;
  const_iterator end() const noexcept;
  const_iterator cbegin() const noexcept;
  const_iterator cend() const noexcept;

  bool operator==(const String &) const noexcept;
  bool operator!=(const String &) const noexcept;
  bool operator<(const String &) const noexcept;
  bool operator<=(const String &) const noexcept;
  bool operator>(const String &) const noexcept;
  bool operator>=(const String &) const noexcept;

  void swap(String &) noexcept;

  String(unsafe_bitcopy_t, const String &) noexcept;

private:
  struct lossy_t;
  String(lossy_t, const char *, std::size_t) noexcept;
  String(lossy_t, const char16_t *, std::size_t) noexcept;
  friend void swap(String &lhs, String &rhs) noexcept { lhs.swap(rhs); }

  std::array<std::uintptr_t, 3> repr;
};
#endif // CXXBRIDGE1_RUST_STRING

#ifndef CXXBRIDGE1_RUST_STR
#define CXXBRIDGE1_RUST_STR
class Str final {
public:
  Str() noexcept;
  Str(const String &) noexcept;
  Str(const std::string &);
  Str(const char *);
  Str(const char *, std::size_t);

  Str &operator=(const Str &) & noexcept = default;

  explicit operator std::string() const;
#if __cplusplus >= 201703L
  explicit operator std::string_view() const;
#endif

  const char *data() const noexcept;
  std::size_t size() const noexcept;
  std::size_t length() const noexcept;
  bool empty() const noexcept;

  Str(const Str &) noexcept = default;
  ~Str() noexcept = default;

  using iterator = const char *;
  using const_iterator = const char *;
  const_iterator begin() const noexcept;
  const_iterator end() const noexcept;
  const_iterator cbegin() const noexcept;
  const_iterator cend() const noexcept;

  bool operator==(const Str &) const noexcept;
  bool operator!=(const Str &) const noexcept;
  bool operator<(const Str &) const noexcept;
  bool operator<=(const Str &) const noexcept;
  bool operator>(const Str &) const noexcept;
  bool operator>=(const Str &) const noexcept;

  void swap(Str &) noexcept;

private:
  class uninit;
  Str(uninit) noexcept;
  friend impl<Str>;

  std::array<std::uintptr_t, 2> repr;
};
#endif // CXXBRIDGE1_RUST_STR

#ifndef CXXBRIDGE1_RUST_SLICE
#define CXXBRIDGE1_RUST_SLICE
namespace detail {
template <bool>
struct copy_assignable_if {};

template <>
struct copy_assignable_if<false> {
  copy_assignable_if() noexcept = default;
  copy_assignable_if(const copy_assignable_if &) noexcept = default;
  copy_assignable_if &operator=(const copy_assignable_if &) & noexcept = delete;
  copy_assignable_if &operator=(copy_assignable_if &&) & noexcept = default;
};
} // namespace detail

template <typename T>
class Slice final
    : private detail::copy_assignable_if<std::is_const<T>::value> {
public:
  using value_type = T;

  Slice() noexcept;
  Slice(T *, std::size_t count) noexcept;

  template <typename C>
  explicit Slice(C &c) : Slice(c.data(), c.size()) {}

  Slice &operator=(const Slice<T> &) & noexcept = default;
  Slice &operator=(Slice<T> &&) & noexcept = default;

  T *data() const noexcept;
  std::size_t size() const noexcept;
  std::size_t length() const noexcept;
  bool empty() const noexcept;

  T &operator[](std::size_t n) const noexcept;
  T &at(std::size_t n) const;
  T &front() const noexcept;
  T &back() const noexcept;

  Slice(const Slice<T> &) noexcept = default;
  ~Slice() noexcept = default;

  class iterator;
  iterator begin() const noexcept;
  iterator end() const noexcept;

  void swap(Slice &) noexcept;

private:
  class uninit;
  Slice(uninit) noexcept;
  friend impl<Slice>;
  friend void sliceInit(void *, const void *, std::size_t) noexcept;
  friend void *slicePtr(const void *) noexcept;
  friend std::size_t sliceLen(const void *) noexcept;

  std::array<std::uintptr_t, 2> repr;
};

#ifdef __cpp_deduction_guides
template <typename C>
explicit Slice(C &c)
    -> Slice<std::remove_reference_t<decltype(*std::declval<C>().data())>>;
#endif // __cpp_deduction_guides

template <typename T>
class Slice<T>::iterator final {
public:
#if __cplusplus >= 202002L
  using iterator_category = std::contiguous_iterator_tag;
#else
  using iterator_category = std::random_access_iterator_tag;
#endif
  using value_type = T;
  using difference_type = std::ptrdiff_t;
  using pointer = typename std::add_pointer<T>::type;
  using reference = typename std::add_lvalue_reference<T>::type;

  reference operator*() const noexcept;
  pointer operator->() const noexcept;
  reference operator[](difference_type) const noexcept;

  iterator &operator++() noexcept;
  iterator operator++(int) noexcept;
  iterator &operator--() noexcept;
  iterator operator--(int) noexcept;

  iterator &operator+=(difference_type) noexcept;
  iterator &operator-=(difference_type) noexcept;
  iterator operator+(difference_type) const noexcept;
  friend inline iterator operator+(difference_type lhs, iterator rhs) noexcept {
    return rhs + lhs;
  }
  iterator operator-(difference_type) const noexcept;
  difference_type operator-(const iterator &) const noexcept;

  bool operator==(const iterator &) const noexcept;
  bool operator!=(const iterator &) const noexcept;
  bool operator<(const iterator &) const noexcept;
  bool operator<=(const iterator &) const noexcept;
  bool operator>(const iterator &) const noexcept;
  bool operator>=(const iterator &) const noexcept;

private:
  friend class Slice;
  void *pos;
  std::size_t stride;
};

#if __cplusplus >= 202002L
static_assert(std::ranges::contiguous_range<rust::Slice<const uint8_t>>);
static_assert(std::contiguous_iterator<rust::Slice<const uint8_t>::iterator>);
#endif

template <typename T>
Slice<T>::Slice() noexcept {
  sliceInit(this, reinterpret_cast<void *>(align_of<T>()), 0);
}

template <typename T>
Slice<T>::Slice(T *s, std::size_t count) noexcept {
  assert(s != nullptr || count == 0);
  sliceInit(this,
            s == nullptr && count == 0
                ? reinterpret_cast<void *>(align_of<T>())
                : const_cast<typename std::remove_const<T>::type *>(s),
            count);
}

template <typename T>
T *Slice<T>::data() const noexcept {
  return reinterpret_cast<T *>(slicePtr(this));
}

template <typename T>
std::size_t Slice<T>::size() const noexcept {
  return sliceLen(this);
}

template <typename T>
std::size_t Slice<T>::length() const noexcept {
  return this->size();
}

template <typename T>
bool Slice<T>::empty() const noexcept {
  return this->size() == 0;
}

template <typename T>
T &Slice<T>::operator[](std::size_t n) const noexcept {
  assert(n < this->size());
  auto ptr = static_cast<char *>(slicePtr(this)) + size_of<T>() * n;
  return *reinterpret_cast<T *>(ptr);
}

template <typename T>
T &Slice<T>::at(std::size_t n) const {
  if (n >= this->size()) {
    panic<std::out_of_range>("rust::Slice index out of range");
  }
  return (*this)[n];
}

template <typename T>
T &Slice<T>::front() const noexcept {
  assert(!this->empty());
  return (*this)[0];
}

template <typename T>
T &Slice<T>::back() const noexcept {
  assert(!this->empty());
  return (*this)[this->size() - 1];
}

template <typename T>
typename Slice<T>::iterator::reference
Slice<T>::iterator::operator*() const noexcept {
  return *static_cast<T *>(this->pos);
}

template <typename T>
typename Slice<T>::iterator::pointer
Slice<T>::iterator::operator->() const noexcept {
  return static_cast<T *>(this->pos);
}

template <typename T>
typename Slice<T>::iterator::reference Slice<T>::iterator::operator[](
    typename Slice<T>::iterator::difference_type n) const noexcept {
  auto ptr = static_cast<char *>(this->pos) + this->stride * n;
  return *reinterpret_cast<T *>(ptr);
}

template <typename T>
typename Slice<T>::iterator &Slice<T>::iterator::operator++() noexcept {
  this->pos = static_cast<char *>(this->pos) + this->stride;
  return *this;
}

template <typename T>
typename Slice<T>::iterator Slice<T>::iterator::operator++(int) noexcept {
  auto ret = iterator(*this);
  this->pos = static_cast<char *>(this->pos) + this->stride;
  return ret;
}

template <typename T>
typename Slice<T>::iterator &Slice<T>::iterator::operator--() noexcept {
  this->pos = static_cast<char *>(this->pos) - this->stride;
  return *this;
}

template <typename T>
typename Slice<T>::iterator Slice<T>::iterator::operator--(int) noexcept {
  auto ret = iterator(*this);
  this->pos = static_cast<char *>(this->pos) - this->stride;
  return ret;
}

template <typename T>
typename Slice<T>::iterator &Slice<T>::iterator::operator+=(
    typename Slice<T>::iterator::difference_type n) noexcept {
  this->pos = static_cast<char *>(this->pos) + this->stride * n;
  return *this;
}

template <typename T>
typename Slice<T>::iterator &Slice<T>::iterator::operator-=(
    typename Slice<T>::iterator::difference_type n) noexcept {
  this->pos = static_cast<char *>(this->pos) - this->stride * n;
  return *this;
}

template <typename T>
typename Slice<T>::iterator Slice<T>::iterator::operator+(
    typename Slice<T>::iterator::difference_type n) const noexcept {
  auto ret = iterator(*this);
  ret.pos = static_cast<char *>(this->pos) + this->stride * n;
  return ret;
}

template <typename T>
typename Slice<T>::iterator Slice<T>::iterator::operator-(
    typename Slice<T>::iterator::difference_type n) const noexcept {
  auto ret = iterator(*this);
  ret.pos = static_cast<char *>(this->pos) - this->stride * n;
  return ret;
}

template <typename T>
typename Slice<T>::iterator::difference_type
Slice<T>::iterator::operator-(const iterator &other) const noexcept {
  auto diff = std::distance(static_cast<char *>(other.pos),
                            static_cast<char *>(this->pos));
  return diff / static_cast<typename Slice<T>::iterator::difference_type>(
                    this->stride);
}

template <typename T>
bool Slice<T>::iterator::operator==(const iterator &other) const noexcept {
  return this->pos == other.pos;
}

template <typename T>
bool Slice<T>::iterator::operator!=(const iterator &other) const noexcept {
  return this->pos != other.pos;
}

template <typename T>
bool Slice<T>::iterator::operator<(const iterator &other) const noexcept {
  return this->pos < other.pos;
}

template <typename T>
bool Slice<T>::iterator::operator<=(const iterator &other) const noexcept {
  return this->pos <= other.pos;
}

template <typename T>
bool Slice<T>::iterator::operator>(const iterator &other) const noexcept {
  return this->pos > other.pos;
}

template <typename T>
bool Slice<T>::iterator::operator>=(const iterator &other) const noexcept {
  return this->pos >= other.pos;
}

template <typename T>
typename Slice<T>::iterator Slice<T>::begin() const noexcept {
  iterator it;
  it.pos = slicePtr(this);
  it.stride = size_of<T>();
  return it;
}

template <typename T>
typename Slice<T>::iterator Slice<T>::end() const noexcept {
  iterator it = this->begin();
  it.pos = static_cast<char *>(it.pos) + it.stride * this->size();
  return it;
}

template <typename T>
void Slice<T>::swap(Slice &rhs) noexcept {
  std::swap(*this, rhs);
}
#endif // CXXBRIDGE1_RUST_SLICE

#ifndef CXXBRIDGE1_RUST_BITCOPY_T
#define CXXBRIDGE1_RUST_BITCOPY_T
struct unsafe_bitcopy_t final {
  explicit unsafe_bitcopy_t() = default;
};
#endif // CXXBRIDGE1_RUST_BITCOPY_T

#ifndef CXXBRIDGE1_RUST_VEC
#define CXXBRIDGE1_RUST_VEC
template <typename T>
class Vec final {
public:
  using value_type = T;

  Vec() noexcept;
  Vec(std::initializer_list<T>);
  Vec(const Vec &);
  Vec(Vec &&) noexcept;
  ~Vec() noexcept;

  Vec &operator=(Vec &&) & noexcept;
  Vec &operator=(const Vec &) &;

  std::size_t size() const noexcept;
  bool empty() const noexcept;
  const T *data() const noexcept;
  T *data() noexcept;
  std::size_t capacity() const noexcept;

  const T &operator[](std::size_t n) const noexcept;
  const T &at(std::size_t n) const;
  const T &front() const noexcept;
  const T &back() const noexcept;

  T &operator[](std::size_t n) noexcept;
  T &at(std::size_t n);
  T &front() noexcept;
  T &back() noexcept;

  void reserve(std::size_t new_cap);
  void push_back(const T &value);
  void push_back(T &&value);
  template <typename... Args>
  void emplace_back(Args &&...args);
  void truncate(std::size_t len);
  void clear();

  using iterator = typename Slice<T>::iterator;
  iterator begin() noexcept;
  iterator end() noexcept;

  using const_iterator = typename Slice<const T>::iterator;
  const_iterator begin() const noexcept;
  const_iterator end() const noexcept;
  const_iterator cbegin() const noexcept;
  const_iterator cend() const noexcept;

  void swap(Vec &) noexcept;

  Vec(unsafe_bitcopy_t, const Vec &) noexcept;

private:
  void reserve_total(std::size_t new_cap) noexcept;
  void set_len(std::size_t len) noexcept;
  void drop() noexcept;

  friend void swap(Vec &lhs, Vec &rhs) noexcept { lhs.swap(rhs); }

  std::array<std::uintptr_t, 3> repr;
};

template <typename T>
Vec<T>::Vec(std::initializer_list<T> init) : Vec{} {
  this->reserve_total(init.size());
  std::move(init.begin(), init.end(), std::back_inserter(*this));
}

template <typename T>
Vec<T>::Vec(const Vec &other) : Vec() {
  this->reserve_total(other.size());
  std::copy(other.begin(), other.end(), std::back_inserter(*this));
}

template <typename T>
Vec<T>::Vec(Vec &&other) noexcept : repr(other.repr) {
  new (&other) Vec();
}

template <typename T>
Vec<T>::~Vec() noexcept {
  this->drop();
}

template <typename T>
Vec<T> &Vec<T>::operator=(Vec &&other) & noexcept {
  this->drop();
  this->repr = other.repr;
  new (&other) Vec();
  return *this;
}

template <typename T>
Vec<T> &Vec<T>::operator=(const Vec &other) & {
  if (this != &other) {
    this->drop();
    new (this) Vec(other);
  }
  return *this;
}

template <typename T>
bool Vec<T>::empty() const noexcept {
  return this->size() == 0;
}

template <typename T>
T *Vec<T>::data() noexcept {
  return const_cast<T *>(const_cast<const Vec<T> *>(this)->data());
}

template <typename T>
const T &Vec<T>::operator[](std::size_t n) const noexcept {
  assert(n < this->size());
  auto data = reinterpret_cast<const char *>(this->data());
  return *reinterpret_cast<const T *>(data + n * size_of<T>());
}

template <typename T>
const T &Vec<T>::at(std::size_t n) const {
  if (n >= this->size()) {
    panic<std::out_of_range>("rust::Vec index out of range");
  }
  return (*this)[n];
}

template <typename T>
const T &Vec<T>::front() const noexcept {
  assert(!this->empty());
  return (*this)[0];
}

template <typename T>
const T &Vec<T>::back() const noexcept {
  assert(!this->empty());
  return (*this)[this->size() - 1];
}

template <typename T>
T &Vec<T>::operator[](std::size_t n) noexcept {
  assert(n < this->size());
  auto data = reinterpret_cast<char *>(this->data());
  return *reinterpret_cast<T *>(data + n * size_of<T>());
}

template <typename T>
T &Vec<T>::at(std::size_t n) {
  if (n >= this->size()) {
    panic<std::out_of_range>("rust::Vec index out of range");
  }
  return (*this)[n];
}

template <typename T>
T &Vec<T>::front() noexcept {
  assert(!this->empty());
  return (*this)[0];
}

template <typename T>
T &Vec<T>::back() noexcept {
  assert(!this->empty());
  return (*this)[this->size() - 1];
}

template <typename T>
void Vec<T>::reserve(std::size_t new_cap) {
  this->reserve_total(new_cap);
}

template <typename T>
void Vec<T>::push_back(const T &value) {
  this->emplace_back(value);
}

template <typename T>
void Vec<T>::push_back(T &&value) {
  this->emplace_back(std::move(value));
}

template <typename T>
template <typename... Args>
void Vec<T>::emplace_back(Args &&...args) {
  auto size = this->size();
  this->reserve_total(size + 1);
  ::new (reinterpret_cast<T *>(reinterpret_cast<char *>(this->data()) +
                               size * size_of<T>()))
      T(std::forward<Args>(args)...);
  this->set_len(size + 1);
}

template <typename T>
void Vec<T>::clear() {
  this->truncate(0);
}

template <typename T>
typename Vec<T>::iterator Vec<T>::begin() noexcept {
  return Slice<T>(this->data(), this->size()).begin();
}

template <typename T>
typename Vec<T>::iterator Vec<T>::end() noexcept {
  return Slice<T>(this->data(), this->size()).end();
}

template <typename T>
typename Vec<T>::const_iterator Vec<T>::begin() const noexcept {
  return this->cbegin();
}

template <typename T>
typename Vec<T>::const_iterator Vec<T>::end() const noexcept {
  return this->cend();
}

template <typename T>
typename Vec<T>::const_iterator Vec<T>::cbegin() const noexcept {
  return Slice<const T>(this->data(), this->size()).begin();
}

template <typename T>
typename Vec<T>::const_iterator Vec<T>::cend() const noexcept {
  return Slice<const T>(this->data(), this->size()).end();
}

template <typename T>
void Vec<T>::swap(Vec &rhs) noexcept {
  using std::swap;
  swap(this->repr, rhs.repr);
}

template <typename T>
Vec<T>::Vec(unsafe_bitcopy_t, const Vec &bits) noexcept : repr(bits.repr) {}
#endif // CXXBRIDGE1_RUST_VEC

#ifndef CXXBRIDGE1_IS_COMPLETE
#define CXXBRIDGE1_IS_COMPLETE
namespace detail {
namespace {
template <typename T, typename = std::size_t>
struct is_complete : std::false_type {};
template <typename T>
struct is_complete<T, decltype(sizeof(T))> : std::true_type {};
} // namespace
} // namespace detail
#endif // CXXBRIDGE1_IS_COMPLETE

#ifndef CXXBRIDGE1_LAYOUT
#define CXXBRIDGE1_LAYOUT
class layout {
  template <typename T>
  friend std::size_t size_of();
  template <typename T>
  friend std::size_t align_of();
  template <typename T>
  static typename std::enable_if<std::is_base_of<Opaque, T>::value,
                                 std::size_t>::type
  do_size_of() {
    return T::layout::size();
  }
  template <typename T>
  static typename std::enable_if<!std::is_base_of<Opaque, T>::value,
                                 std::size_t>::type
  do_size_of() {
    return sizeof(T);
  }
  template <typename T>
  static
      typename std::enable_if<detail::is_complete<T>::value, std::size_t>::type
      size_of() {
    return do_size_of<T>();
  }
  template <typename T>
  static typename std::enable_if<std::is_base_of<Opaque, T>::value,
                                 std::size_t>::type
  do_align_of() {
    return T::layout::align();
  }
  template <typename T>
  static typename std::enable_if<!std::is_base_of<Opaque, T>::value,
                                 std::size_t>::type
  do_align_of() {
    return alignof(T);
  }
  template <typename T>
  static
      typename std::enable_if<detail::is_complete<T>::value, std::size_t>::type
      align_of() {
    return do_align_of<T>();
  }
};

template <typename T>
std::size_t size_of() {
  return layout::size_of<T>();
}

template <typename T>
std::size_t align_of() {
  return layout::align_of<T>();
}
#endif // CXXBRIDGE1_LAYOUT

namespace {
template <bool> struct deleter_if {
  template <typename T> void operator()(T *) {}
};
template <> struct deleter_if<true> {
  template <typename T> void operator()(T *ptr) { ptr->~T(); }
};

template <typename T, bool = ::rust::detail::is_complete<T>::value>
struct is_destructible : ::std::false_type {};
template <typename T>
struct is_destructible<T, true> : ::std::is_destructible<T> {};
template <typename T>
struct is_destructible<T[], false> : is_destructible<T> {};
template <typename T, bool = ::rust::is_destructible<T>::value>
struct shared_ptr_if_destructible {
  explicit shared_ptr_if_destructible(typename ::std::shared_ptr<T>::element_type *) {}
};
template <typename T>
struct shared_ptr_if_destructible<T, true> : ::std::shared_ptr<T> {
  using ::std::shared_ptr<T>::shared_ptr;
};
} // namespace
} // namespace cxxbridge1
} // namespace rust

namespace pdns {
  namespace rust {
    namespace misc {
      enum class LogLevel : ::std::uint8_t;
      enum class Priority : ::std::uint8_t;
      struct KeyValue;
      using NetmaskGroup = ::pdns::rust::misc::NetmaskGroup;
      using ComboAddress = ::pdns::rust::misc::ComboAddress;
      using Logger = ::pdns::rust::misc::Logger;
    }
  }
}

namespace pdns {
namespace rust {
namespace misc {
#ifndef CXXBRIDGE1_ENUM_pdns$rust$misc$LogLevel
#define CXXBRIDGE1_ENUM_pdns$rust$misc$LogLevel
enum class LogLevel : ::std::uint8_t {
  None = 0,
  Normal = 1,
  Detailed = 2,
};
#endif // CXXBRIDGE1_ENUM_pdns$rust$misc$LogLevel

#ifndef CXXBRIDGE1_ENUM_pdns$rust$misc$Priority
#define CXXBRIDGE1_ENUM_pdns$rust$misc$Priority
enum class Priority : ::std::uint8_t {
  Absent = 0,
  Alert = 1,
  Critical = 2,
  Error = 3,
  Warning = 4,
  Notice = 5,
  Info = 6,
  Debug = 7,
};
#endif // CXXBRIDGE1_ENUM_pdns$rust$misc$Priority

#ifndef CXXBRIDGE1_STRUCT_pdns$rust$misc$KeyValue
#define CXXBRIDGE1_STRUCT_pdns$rust$misc$KeyValue
struct KeyValue final {
  ::rust::String key;
  ::rust::String value;

  using IsRelocatable = ::std::true_type;
};
#endif // CXXBRIDGE1_STRUCT_pdns$rust$misc$KeyValue

extern "C" {
::std::uint16_t pdns$rust$misc$cxxbridge1$192$qTypeStringToCode(::rust::Str name) noexcept {
  ::std::uint16_t (*qTypeStringToCode$)(::rust::Str) = ::pdns::rust::misc::qTypeStringToCode;
  return qTypeStringToCode$(name);
}

bool pdns$rust$misc$cxxbridge1$192$isValidHostname(::rust::Str name) noexcept {
  bool (*isValidHostname$)(::rust::Str) = ::pdns::rust::misc::isValidHostname;
  return isValidHostname$(name);
}

::pdns::rust::misc::ComboAddress *pdns$rust$misc$cxxbridge1$192$comboaddress(::rust::Str address) noexcept {
  ::std::unique_ptr<::pdns::rust::misc::ComboAddress> (*comboaddress$)(::rust::Str) = ::pdns::rust::misc::comboaddress;
  return comboaddress$(address).release();
}

bool pdns$rust$misc$cxxbridge1$192$matches(::std::unique_ptr<::pdns::rust::misc::NetmaskGroup> const &nmg, ::std::unique_ptr<::pdns::rust::misc::ComboAddress> const &address) noexcept {
  bool (*matches$)(::std::unique_ptr<::pdns::rust::misc::NetmaskGroup> const &, ::std::unique_ptr<::pdns::rust::misc::ComboAddress> const &) = ::pdns::rust::misc::matches;
  return matches$(nmg, address);
}

void pdns$rust$misc$cxxbridge1$192$withValue(::std::shared_ptr<::pdns::rust::misc::Logger> const &logger, ::rust::Str key, ::rust::Str val, ::std::shared_ptr<::pdns::rust::misc::Logger> *return$) noexcept {
  ::std::shared_ptr<::pdns::rust::misc::Logger> (*withValue$)(::std::shared_ptr<::pdns::rust::misc::Logger> const &, ::rust::Str, ::rust::Str) = ::pdns::rust::misc::withValue;
  new (return$) ::std::shared_ptr<::pdns::rust::misc::Logger>(withValue$(logger, key, val));
}

void pdns$rust$misc$cxxbridge1$192$log(::std::shared_ptr<::pdns::rust::misc::Logger> const &logger, ::pdns::rust::misc::Priority prio, ::rust::Str msg, ::rust::Vec<::pdns::rust::misc::KeyValue> const &values) noexcept {
  void (*log$)(::std::shared_ptr<::pdns::rust::misc::Logger> const &, ::pdns::rust::misc::Priority, ::rust::Str, ::rust::Vec<::pdns::rust::misc::KeyValue> const &) = ::pdns::rust::misc::log;
  log$(logger, prio, msg, values);
}

void pdns$rust$misc$cxxbridge1$192$error(::std::shared_ptr<::pdns::rust::misc::Logger> const &logger, ::pdns::rust::misc::Priority prio, ::rust::Str err, ::rust::Str msg, ::rust::Vec<::pdns::rust::misc::KeyValue> const &values) noexcept {
  void (*error$)(::std::shared_ptr<::pdns::rust::misc::Logger> const &, ::pdns::rust::misc::Priority, ::rust::Str, ::rust::Str, ::rust::Vec<::pdns::rust::misc::KeyValue> const &) = ::pdns::rust::misc::error;
  error$(logger, prio, err, msg, values);
}

void pdns$rust$misc$cxxbridge1$192$getUUID(::rust::String *return$) noexcept {
  ::rust::String (*getUUID$)() = ::pdns::rust::misc::getUUID;
  new (return$) ::rust::String(getUUID$());
}
} // extern "C"
} // namespace misc
} // namespace rust
} // namespace pdns

extern "C" {
static_assert(::rust::detail::is_complete<::std::remove_extent<::pdns::rust::misc::ComboAddress>::type>::value, "definition of `::pdns::rust::misc::ComboAddress` is required");
static_assert(sizeof(::std::unique_ptr<::pdns::rust::misc::ComboAddress>) == sizeof(void *), "");
static_assert(alignof(::std::unique_ptr<::pdns::rust::misc::ComboAddress>) == alignof(void *), "");
void cxxbridge1$unique_ptr$pdns$rust$misc$ComboAddress$null(::std::unique_ptr<::pdns::rust::misc::ComboAddress> *ptr) noexcept {
  ::new (ptr) ::std::unique_ptr<::pdns::rust::misc::ComboAddress>();
}
void cxxbridge1$unique_ptr$pdns$rust$misc$ComboAddress$raw(::std::unique_ptr<::pdns::rust::misc::ComboAddress> *ptr, ::std::unique_ptr<::pdns::rust::misc::ComboAddress>::pointer raw) noexcept {
  ::new (ptr) ::std::unique_ptr<::pdns::rust::misc::ComboAddress>(raw);
}
::std::unique_ptr<::pdns::rust::misc::ComboAddress>::element_type const *cxxbridge1$unique_ptr$pdns$rust$misc$ComboAddress$get(::std::unique_ptr<::pdns::rust::misc::ComboAddress> const &ptr) noexcept {
  return ptr.get();
}
::std::unique_ptr<::pdns::rust::misc::ComboAddress>::pointer cxxbridge1$unique_ptr$pdns$rust$misc$ComboAddress$release(::std::unique_ptr<::pdns::rust::misc::ComboAddress> &ptr) noexcept {
  return ptr.release();
}
void cxxbridge1$unique_ptr$pdns$rust$misc$ComboAddress$drop(::std::unique_ptr<::pdns::rust::misc::ComboAddress> *ptr) noexcept {
  ::rust::deleter_if<::rust::detail::is_complete<::pdns::rust::misc::ComboAddress>::value>{}(ptr);
}

static_assert(::rust::detail::is_complete<::std::remove_extent<::pdns::rust::misc::NetmaskGroup>::type>::value, "definition of `::pdns::rust::misc::NetmaskGroup` is required");
static_assert(sizeof(::std::unique_ptr<::pdns::rust::misc::NetmaskGroup>) == sizeof(void *), "");
static_assert(alignof(::std::unique_ptr<::pdns::rust::misc::NetmaskGroup>) == alignof(void *), "");
void cxxbridge1$unique_ptr$pdns$rust$misc$NetmaskGroup$null(::std::unique_ptr<::pdns::rust::misc::NetmaskGroup> *ptr) noexcept {
  ::new (ptr) ::std::unique_ptr<::pdns::rust::misc::NetmaskGroup>();
}
void cxxbridge1$unique_ptr$pdns$rust$misc$NetmaskGroup$raw(::std::unique_ptr<::pdns::rust::misc::NetmaskGroup> *ptr, ::std::unique_ptr<::pdns::rust::misc::NetmaskGroup>::pointer raw) noexcept {
  ::new (ptr) ::std::unique_ptr<::pdns::rust::misc::NetmaskGroup>(raw);
}
::std::unique_ptr<::pdns::rust::misc::NetmaskGroup>::element_type const *cxxbridge1$unique_ptr$pdns$rust$misc$NetmaskGroup$get(::std::unique_ptr<::pdns::rust::misc::NetmaskGroup> const &ptr) noexcept {
  return ptr.get();
}
::std::unique_ptr<::pdns::rust::misc::NetmaskGroup>::pointer cxxbridge1$unique_ptr$pdns$rust$misc$NetmaskGroup$release(::std::unique_ptr<::pdns::rust::misc::NetmaskGroup> &ptr) noexcept {
  return ptr.release();
}
void cxxbridge1$unique_ptr$pdns$rust$misc$NetmaskGroup$drop(::std::unique_ptr<::pdns::rust::misc::NetmaskGroup> *ptr) noexcept {
  ::rust::deleter_if<::rust::detail::is_complete<::pdns::rust::misc::NetmaskGroup>::value>{}(ptr);
}

static_assert(sizeof(::std::shared_ptr<::pdns::rust::misc::Logger>) == 2 * sizeof(void *), "");
static_assert(alignof(::std::shared_ptr<::pdns::rust::misc::Logger>) == alignof(void *), "");
void cxxbridge1$shared_ptr$pdns$rust$misc$Logger$null(::std::shared_ptr<::pdns::rust::misc::Logger> *ptr) noexcept {
  ::new (ptr) ::std::shared_ptr<::pdns::rust::misc::Logger>();
}
bool cxxbridge1$shared_ptr$pdns$rust$misc$Logger$raw(::std::shared_ptr<::pdns::rust::misc::Logger> *ptr, ::std::shared_ptr<::pdns::rust::misc::Logger>::element_type *raw) noexcept {
  ::new (ptr) ::rust::shared_ptr_if_destructible<::pdns::rust::misc::Logger>(raw);
  return ::rust::is_destructible<::pdns::rust::misc::Logger>::value;
}
void cxxbridge1$shared_ptr$pdns$rust$misc$Logger$clone(::std::shared_ptr<::pdns::rust::misc::Logger> const &self, ::std::shared_ptr<::pdns::rust::misc::Logger> *ptr) noexcept {
  ::new (ptr) ::std::shared_ptr<::pdns::rust::misc::Logger>(self);
}
::std::shared_ptr<::pdns::rust::misc::Logger>::element_type const *cxxbridge1$shared_ptr$pdns$rust$misc$Logger$get(::std::shared_ptr<::pdns::rust::misc::Logger> const &self) noexcept {
  return self.get();
}
void cxxbridge1$shared_ptr$pdns$rust$misc$Logger$drop(::std::shared_ptr<::pdns::rust::misc::Logger> *self) noexcept {
  self->~shared_ptr();
}

void cxxbridge1$rust_vec$pdns$rust$misc$KeyValue$new(::rust::Vec<::pdns::rust::misc::KeyValue> const *ptr) noexcept;
void cxxbridge1$rust_vec$pdns$rust$misc$KeyValue$drop(::rust::Vec<::pdns::rust::misc::KeyValue> *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$pdns$rust$misc$KeyValue$len(::rust::Vec<::pdns::rust::misc::KeyValue> const *ptr) noexcept;
::std::size_t cxxbridge1$rust_vec$pdns$rust$misc$KeyValue$capacity(::rust::Vec<::pdns::rust::misc::KeyValue> const *ptr) noexcept;
::pdns::rust::misc::KeyValue const *cxxbridge1$rust_vec$pdns$rust$misc$KeyValue$data(::rust::Vec<::pdns::rust::misc::KeyValue> const *ptr) noexcept;
void cxxbridge1$rust_vec$pdns$rust$misc$KeyValue$reserve_total(::rust::Vec<::pdns::rust::misc::KeyValue> *ptr, ::std::size_t new_cap) noexcept;
void cxxbridge1$rust_vec$pdns$rust$misc$KeyValue$set_len(::rust::Vec<::pdns::rust::misc::KeyValue> *ptr, ::std::size_t len) noexcept;
void cxxbridge1$rust_vec$pdns$rust$misc$KeyValue$truncate(::rust::Vec<::pdns::rust::misc::KeyValue> *ptr, ::std::size_t len) noexcept;
} // extern "C"

namespace rust {
inline namespace cxxbridge1 {
template <>
Vec<::pdns::rust::misc::KeyValue>::Vec() noexcept {
  cxxbridge1$rust_vec$pdns$rust$misc$KeyValue$new(this);
}
template <>
void Vec<::pdns::rust::misc::KeyValue>::drop() noexcept {
  return cxxbridge1$rust_vec$pdns$rust$misc$KeyValue$drop(this);
}
template <>
::std::size_t Vec<::pdns::rust::misc::KeyValue>::size() const noexcept {
  return cxxbridge1$rust_vec$pdns$rust$misc$KeyValue$len(this);
}
template <>
::std::size_t Vec<::pdns::rust::misc::KeyValue>::capacity() const noexcept {
  return cxxbridge1$rust_vec$pdns$rust$misc$KeyValue$capacity(this);
}
template <>
::pdns::rust::misc::KeyValue const *Vec<::pdns::rust::misc::KeyValue>::data() const noexcept {
  return cxxbridge1$rust_vec$pdns$rust$misc$KeyValue$data(this);
}
template <>
void Vec<::pdns::rust::misc::KeyValue>::reserve_total(::std::size_t new_cap) noexcept {
  return cxxbridge1$rust_vec$pdns$rust$misc$KeyValue$reserve_total(this, new_cap);
}
template <>
void Vec<::pdns::rust::misc::KeyValue>::set_len(::std::size_t len) noexcept {
  return cxxbridge1$rust_vec$pdns$rust$misc$KeyValue$set_len(this, len);
}
template <>
void Vec<::pdns::rust::misc::KeyValue>::truncate(::std::size_t len) {
  return cxxbridge1$rust_vec$pdns$rust$misc$KeyValue$truncate(this, len);
}
} // namespace cxxbridge1
} // namespace rust
