// Copyright 2019 Drone IO, Inc.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package status

import (
	"context"
	"fmt"

	"github.com/drone/drone/core"
	"github.com/drone/go-scm/scm"
	"github.com/drone/go-scm/scm/driver/github"
)

// Config configures the Status service.
type Config struct {
	Base     string
	Name     string
	Disabled bool
}

// New returns a new StatusService
func New(client *scm.Client, renew core.Renewer, config Config) core.StatusService {
	return &service{
		client:   client,
		renew:    renew,
		base:     config.Base,
		name:     config.Name,
		disabled: config.Disabled,
	}
}

type service struct {
	renew    core.Renewer
	client   *scm.Client
	base     string
	name     string
	disabled bool
}

func (s *service) Send(ctx context.Context, user *core.User, req *core.StatusInput) error {
	if s.disabled || req.Build.Event == core.EventCron {
		return nil
	}

	err := s.renew.Renew(ctx, user, false)
	if err != nil {
		return err
	}

	ctx = context.WithValue(ctx, scm.TokenKey{}, &scm.Token{
		Token:   user.Token,
		Refresh: user.Refresh,
	})

	// HACK(bradrydzewski) provides support for the github deployment API
	if req.Build.DeployID != 0 && s.client.Driver == scm.DriverGithub {
		// TODO(bradrydzewski) only update the deployment status when the
		// build completes.
		if req.Build.Finished == 0 {
			return nil
		}
		_, _, err = s.client.Repositories.(*github.RepositoryService).CreateDeployStatus(ctx, req.Repo.Slug, &scm.DeployStatus{
			Number:      req.Build.DeployID,
			Desc:        createDesc(req.Build.Status),
			State:       convertStatus(req.Build.Status),
			Target:      fmt.Sprintf("%s/%s/%d", s.base, req.Repo.Slug, req.Build.Number),
			Environment: req.Build.Deploy,
		})
		return err
	}

	_, _, err = s.client.Repositories.CreateStatus(ctx, req.Repo.Slug, req.Build.After, &scm.StatusInput{
		Title:  fmt.Sprintf("Build #%d", req.Build.Number),
		Desc:   createDesc(req.Build.Status),
		Label:  createLabel(s.name, req.Build.Event, req.Build.Deploy),
		State:  convertStatus(req.Build.Status),
		Target: fmt.Sprintf("%s/%s/%d", s.base, req.Repo.Slug, req.Build.Number),
	})
	if err == scm.ErrNotSupported {
		return nil
	}
	return err
}
