// Copyright 2019 Drone IO, Inc.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package web

import (
	"net/http"
	"time"

	"github.com/drone/drone/core"
	"github.com/drone/go-scm/scm"
)

type varz struct {
	SCM     *scmInfo     `json:"scm"`
	License *licenseInfo `json:"license"`
}

type scmInfo struct {
	URL  string    `json:"url"`
	Rate *rateInfo `json:"rate"`
}

type rateInfo struct {
	Limit     int   `json:"limit"`
	Remaining int   `json:"remaining"`
	Reset     int64 `json:"reset"`
}

type licenseInfo struct {
	Kind       string    `json:"kind"`
	Seats      int64     `json:"seats"`
	SeatsUsed  int64     `json:"seats_used,omitempty"`
	SeatsAvail int64     `json:"seats_available,omitempty"`
	Repos      int64     `json:"repos"`
	ReposUsed  int64     `json:"repos_used,omitempty"`
	ReposAvail int64     `json:"repos_available,omitempty"`
	Expires    time.Time `json:"expire_at,omitempty"`
}

// HandleVarz creates an http.HandlerFunc that exposes internal system
// information.
func HandleVarz(client *scm.Client, license *core.License) http.HandlerFunc {
	return func(w http.ResponseWriter, r *http.Request) {
		rate := client.Rate()
		v := &varz{
			License: &licenseInfo{
				Kind:    license.Kind,
				Seats:   license.Users,
				Repos:   license.Repos,
				Expires: license.Expires,
			},
			SCM: &scmInfo{
				URL: client.BaseURL.String(),
				Rate: &rateInfo{
					Limit:     rate.Limit,
					Remaining: rate.Remaining,
					Reset:     rate.Reset,
				},
			},
		}
		writeJSON(w, v, 200)
	}
}
