/*
Copyright 2020 The cert-manager Authors.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

    http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License.
*/

package v1

// ConditionStatus represents a condition's status.
// +kubebuilder:validation:Enum=True;False;Unknown
type ConditionStatus string

// These are valid condition statuses. "ConditionTrue" means a resource is in
// the condition; "ConditionFalse" means a resource is not in the condition;
// "ConditionUnknown" means kubernetes can't decide if a resource is in the
// condition or not. In the future, we could add other intermediate
// conditions, e.g., ConditionDegraded.
const (
	// ConditionTrue represents the fact that a given condition is true
	ConditionTrue ConditionStatus = "True"

	// ConditionFalse represents the fact that a given condition is false
	ConditionFalse ConditionStatus = "False"

	// ConditionUnknown represents the fact that a given condition is unknown
	ConditionUnknown ConditionStatus = "Unknown"
)

// A reference to an object in the same namespace as the referent.
// If the referent is a cluster-scoped resource (e.g., a ClusterIssuer),
// the reference instead refers to the resource with the given name in the
// configured 'cluster resource namespace', which is set as a flag on the
// controller component (and defaults to the namespace that cert-manager
// runs in).
type LocalObjectReference struct {
	// Name of the resource being referred to.
	// More info: https://kubernetes.io/docs/concepts/overview/working-with-objects/names/#names
	Name string `json:"name"`
}

// IssuerReference is a reference to a certificate issuer object with a given name, kind and group.
type IssuerReference struct {
	// Name of the issuer being referred to.
	Name string `json:"name"`
	// Kind of the issuer being referred to.
	// Defaults to 'Issuer'.
	// +optional
	Kind string `json:"kind,omitempty"`
	// Group of the issuer being referred to.
	// Defaults to 'cert-manager.io'.
	// +optional
	Group string `json:"group,omitempty"`
}

// ObjectReference is a reference to an object with a given name, kind and group.
// Deprecated: Use IssuerReference instead.
type ObjectReference = IssuerReference

// A reference to a specific 'key' within a Secret resource.
// In some instances, `key` is a required field.
type SecretKeySelector struct {
	// The name of the Secret resource being referred to.
	LocalObjectReference `json:",inline"`

	// The key of the entry in the Secret resource's `data` field to be used.
	// Some instances of this field may be defaulted, in others it may be
	// required.
	// +optional
	Key string `json:"key,omitempty"`
}

const (
	// Used as a data key in Secret resources to store a CA certificate.
	TLSCAKey = "ca.crt"
)
