/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */
#include <aws/checksums/crc.h>
#include <aws/checksums/private/crc32_priv.h>
#include <aws/checksums/private/crc_util.h>

#include <aws/common/cpuid.h>

large_buffer_apply_impl(crc32, uint32_t)

    AWS_ALIGNED_TYPEDEF(aws_checksums_crc32_constants_t, checksums_constants, 16);

checksums_constants aws_checksums_crc32_constants = {
    .mu_poly = {0x0f7011641, 0x01db710641},
    .shift_factors = {// Position 0
                      {{0x0000000000000000, 0x0000000000000000},
                       {0x0000000077073096, 0x0100000000000000},
                       {0x00000000191b3141, 0x0001000000000000},
                       {0x0000000001c26a37, 0x0000010000000000},
                       {0x00000000b8bc6765, 0x0000000100000000},
                       {0x000000003d6029b0, 0x7707309600000000},
                       {0x00000000cb5cd3a5, 0x191b314100000000},
                       {0x00000000a6770bb4, 0x01c26a3700000000},
                       {0x00000000ccaa009e, 0xb8bc676500000000},
                       {0x00000000177b1443, 0x3d6029b000000000},
                       {0x00000000efc26b3e, 0xcb5cd3a500000000},
                       {0x00000000c18edfc0, 0xa6770bb400000000},
                       {0x000000009ba54c6f, 0xccaa009e00000000},
                       {0x00000000dd96d985, 0x177b144300000000},
                       {0x000000009d0fe176, 0xefc26b3e00000000},
                       {0x00000000b9fbdbe8, 0xc18edfc000000000}},
                      // Position 1
                      {{0x0000000000000000, 0x0000000000000000},
                       {0x00000000ae689191, 0x9ba54c6f00000000},
                       {0x00000000f1da05aa, 0x01b5fd1d00000000},
                       {0x000000003db1ecdc, 0x2a28386200000000},
                       {0x000000008f352d95, 0xcad38e8f00000000},
                       {0x000000001c279815, 0x8e42b13e00000000},
                       {0x00000000df068dc2, 0x101a233100000000},
                       {0x0000000031f8303f, 0xc64ac0b800000000},
                       {0x0000000033fff533, 0x7406fa9500000000},
                       {0x0000000026b70c3d, 0x6dd804d900000000},
                       {0x00000000e3543be0, 0x1d5dce4400000000},
                       {0x000000005a1bb05d, 0xf09a54ac00000000},
                       {0x00000000596c8d81, 0xc56d949600000000},
                       {0x00000000682bdd4f, 0x523d48c400000000},
                       {0x000000004a28bd43, 0xd63a56a600000000},
                       {0x000000000077f00d, 0xc4d49c3900000000}},
                      // Position 2
                      {{0x0000000000000000, 0x0000000000000000},
                       {0x00000000ce3371cb, 0x03f9f86300000000},
                       {0x000000001072db28, 0xc78c44a100000000},
                       {0x000000001423c53a, 0x0ac2d90400000000},
                       {0x00000000f891f16f, 0xc84f001c00000000},
                       {0x00000000b8b4266a, 0xbdf081c400000000},
                       {0x00000000ca073d7f, 0xa55318bd00000000},
                       {0x000000007a23ac44, 0x231f091c00000000},
                       {0x0000000088b6ba63, 0x9a8f75c000000000},
                       {0x000000005a50a058, 0x8be6bf4900000000},
                       {0x0000000003a06621, 0x16335cfe00000000},
                       {0x0000000006c75cad, 0x85dd353900000000},
                       {0x00000000b5d515b6, 0xc68f497a00000000},
                       {0x000000007b01e242, 0x694ce1d700000000},
                       {0x00000000b05e8440, 0x27a81a8400000000},
                       {0x00000000e9b2e84c, 0xb6d9b4e400000000}},
                      // Position 3
                      {{0x0000000000000000, 0x0000000000000000},
                       {0x000000005ad8a92c, 0x13fca91e00000000},
                       {0x00000000b566f6e2, 0xdc7b5c5f00000000},
                       {0x000000008c7515be, 0x121b8a8b00000000},
                       {0x000000009d9129bf, 0x606c5e3400000000},
                       {0x00000000e6c70e63, 0xcd028bc200000000},
                       {0x00000000572f9e69, 0x3f9def1b00000000},
                       {0x00000000530581b4, 0xaae2f4d700000000},
                       {0x000000002ee43a2c, 0xf6b5398600000000},
                       {0x00000000066ff6b1, 0xd7bf8f2500000000},
                       {0x00000000e7508e6d, 0x485d433300000000},
                       {0x00000000dc237963, 0x84a316e900000000},
                       {0x00000000e395e5dc, 0x0397bf9500000000},
                       {0x00000000b3db0f7f, 0x04a2a88300000000},
                       {0x0000000043af81df, 0x1f3e000400000000},
                       {0x000000000f7dc0ea, 0x3b59667d00000000}},
                      // Position 4
                      {{0x0000000000000000, 0x0000000000000000},
                       {0x000000001b4511ee, 0x63fd82d200000000},
                       {0x00000000bce15202, 0xe4a9021500000000},
                       {0x00000000f78d5136, 0x2d30216a00000000},
                       {0x00000000ff08e5ef, 0xd91bdb8800000000},
                       {0x00000000dbeda4f7, 0x4909e37800000000},
                       {0x00000000a29c6b48, 0x157e349e00000000},
                       {0x00000000bb7405e6, 0x6395018400000000},
                       {0x000000006f76172e, 0x2bad0e9a00000000},
                       {0x000000003e2af045, 0xa125c29f00000000},
                       {0x000000005658d844, 0x80c9584f00000000},
                       {0x0000000064711657, 0xce6c873800000000},
                       {0x00000000827ae4f2, 0x0e8e8c8800000000},
                       {0x00000000ea938168, 0x335da4cd00000000},
                       {0x000000000dbc60c2, 0x2d9daa0f00000000},
                       {0x000000001e1255dd, 0x1e84ee2100000000}},
                      // Position 5
                      {{0x0000000000000000, 0x0000000000000000},
                       {0x0000000056f5cab9, 0xbfbcf49c00000000},
                       {0x00000000385993ac, 0xaec31a2f00000000},
                       {0x00000000a3d2cfe4, 0xf33519ee00000000},
                       {0x00000000b4247b20, 0x5c9cbdde00000000},
                       {0x00000000aee87806, 0xf5a4ad9800000000},
                       {0x00000000778a63ce, 0xe4ba555a00000000},
                       {0x00000000adcc94f4, 0x5df0e43d00000000},
                       {0x00000000695186a7, 0x9d57442800000000},
                       {0x0000000075cdb7ed, 0x0bdf6d5a00000000},
                       {0x00000000ce126d13, 0x6d4c124600000000},
                       {0x00000000c60d3fe6, 0xcb6f1f1a00000000},
                       {0x00000000bd97339c, 0x64eb408b00000000},
                       {0x00000000a65f7e5e, 0xaa4a448000000000},
                       {0x00000000a69617d3, 0x42ef789000000000},
                       {0x000000008d175638, 0x688d909000000000}},
                      // Position 6
                      {{0x0000000000000000, 0x0000000000000000},
                       {0x0000000066bc001e, 0x8a39e3c100000000},
                       {0x00000000b58b27b3, 0x85352d3c00000000},
                       {0x000000001acfdbfb, 0x7d0aaeb000000000},
                       {0x00000000fe273162, 0x291a605400000000},
                       {0x00000000a69a8eec, 0x027703b500000000},
                       {0x00000000f654a922, 0xcc6652b600000000},
                       {0x00000000b7620bb0, 0xe0dff1b100000000},
                       {0x00000000d3c98813, 0x53664be100000000},
                       {0x00000000da7a376f, 0x5b3d509700000000},
                       {0x00000000efa9f9ea, 0xe3879e8100000000},
                       {0x00000000a33b0ae3, 0x90a6d47c00000000},
                       {0x000000003d575eef, 0xf7a67e0100000000},
                       {0x0000000025f9e373, 0x428ed66200000000},
                       {0x000000001c9de7a4, 0x98f7ed2100000000},
                       {0x000000007660fe68, 0x67e5af6a00000000}},
                      // Position 7
                      {{0x0000000000000000, 0x0000000000000000},
                       {0x00000000f7d6deb4, 0x52b55e3900000000},
                       {0x00000000edb88320, 0x8000000000000000},
                       {0x000000007beb6f5a, 0xc4e22c3c00000000},
                       {0x0000000076dc4190, 0x4000000000000000},
                       {0x000000003df5b7ad, 0x6271161e00000000},
                       {0x000000003b6e20c8, 0x2000000000000000},
                       {0x00000000f34258f6, 0x31388b0f00000000},
                       {0x000000001db71064, 0x1000000000000000},
                       {0x0000000079a12c7b, 0xf524c6a700000000},
                       {0x000000000edb8832, 0x0800000000000000},
                       {0x00000000d168151d, 0x972ae07300000000},
                       {0x00000000076dc419, 0x0400000000000000},
                       {0x00000000850c89ae, 0xa62df31900000000},
                       {0x00000000ee0e612c, 0x0200000000000000},
                       {0x00000000428644d7, 0xbeae7aac00000000}},
                      // Position 8
                      {{0x0000000000000000, 0x0000000000000000},
                       {0x0000000077073096, 0x0100000000000000},
                       {0x00000000191b3141, 0x0001000000000000},
                       {0x0000000001c26a37, 0x0000010000000000},
                       {0x00000000b8bc6765, 0x0000000100000000},
                       {0x000000003d6029b0, 0x7707309600000000},
                       {0x00000000cb5cd3a5, 0x191b314100000000},
                       {0x00000000a6770bb4, 0x01c26a3700000000},
                       {0x00000000ccaa009e, 0xb8bc676500000000},
                       {0x00000000177b1443, 0x3d6029b000000000},
                       {0x00000000efc26b3e, 0xcb5cd3a500000000},
                       {0x00000000c18edfc0, 0xa6770bb400000000},
                       {0x000000009ba54c6f, 0xccaa009e00000000},
                       {0x00000000dd96d985, 0x177b144300000000},
                       {0x000000009d0fe176, 0xefc26b3e00000000},
                       {0x00000000b9fbdbe8, 0xc18edfc000000000}},
                      // Position 9
                      {{0x0000000000000000, 0x0000000000000000},
                       {0x00000000ae689191, 0x9ba54c6f00000000},
                       {0x00000000f1da05aa, 0x01b5fd1d00000000},
                       {0x000000003db1ecdc, 0x2a28386200000000},
                       {0x000000008f352d95, 0xcad38e8f00000000},
                       {0x000000001c279815, 0x8e42b13e00000000},
                       {0x00000000df068dc2, 0x101a233100000000},
                       {0x0000000031f8303f, 0xc64ac0b800000000},
                       {0x0000000033fff533, 0x7406fa9500000000},
                       {0x0000000026b70c3d, 0x6dd804d900000000},
                       {0x00000000e3543be0, 0x1d5dce4400000000},
                       {0x000000005a1bb05d, 0xf09a54ac00000000},
                       {0x00000000596c8d81, 0xc56d949600000000},
                       {0x00000000682bdd4f, 0x523d48c400000000},
                       {0x000000004a28bd43, 0xd63a56a600000000},
                       {0x000000000077f00d, 0xc4d49c3900000000}},
                      // Position 10
                      {{0x0000000000000000, 0x0000000000000000},
                       {0x00000000ce3371cb, 0x03f9f86300000000},
                       {0x000000001072db28, 0xc78c44a100000000},
                       {0x000000001423c53a, 0x0ac2d90400000000},
                       {0x00000000f891f16f, 0xc84f001c00000000},
                       {0x00000000b8b4266a, 0xbdf081c400000000},
                       {0x00000000ca073d7f, 0xa55318bd00000000},
                       {0x000000007a23ac44, 0x231f091c00000000},
                       {0x0000000088b6ba63, 0x9a8f75c000000000},
                       {0x000000005a50a058, 0x8be6bf4900000000},
                       {0x0000000003a06621, 0x16335cfe00000000},
                       {0x0000000006c75cad, 0x85dd353900000000},
                       {0x00000000b5d515b6, 0xc68f497a00000000},
                       {0x000000007b01e242, 0x694ce1d700000000},
                       {0x00000000b05e8440, 0x27a81a8400000000},
                       {0x00000000e9b2e84c, 0xb6d9b4e400000000}},
                      // Position 11
                      {{0x0000000000000000, 0x0000000000000000},
                       {0x000000005ad8a92c, 0x13fca91e00000000},
                       {0x00000000b566f6e2, 0xdc7b5c5f00000000},
                       {0x000000008c7515be, 0x121b8a8b00000000},
                       {0x000000009d9129bf, 0x606c5e3400000000},
                       {0x00000000e6c70e63, 0xcd028bc200000000},
                       {0x00000000572f9e69, 0x3f9def1b00000000},
                       {0x00000000530581b4, 0xaae2f4d700000000},
                       {0x000000002ee43a2c, 0xf6b5398600000000},
                       {0x00000000066ff6b1, 0xd7bf8f2500000000},
                       {0x00000000e7508e6d, 0x485d433300000000},
                       {0x00000000dc237963, 0x84a316e900000000},
                       {0x00000000e395e5dc, 0x0397bf9500000000},
                       {0x00000000b3db0f7f, 0x04a2a88300000000},
                       {0x0000000043af81df, 0x1f3e000400000000},
                       {0x000000000f7dc0ea, 0x3b59667d00000000}},
                      // Position 12
                      {{0x0000000000000000, 0x0000000000000000},
                       {0x000000001b4511ee, 0x63fd82d200000000},
                       {0x00000000bce15202, 0xe4a9021500000000},
                       {0x00000000f78d5136, 0x2d30216a00000000},
                       {0x00000000ff08e5ef, 0xd91bdb8800000000},
                       {0x00000000dbeda4f7, 0x4909e37800000000},
                       {0x00000000a29c6b48, 0x157e349e00000000},
                       {0x00000000bb7405e6, 0x6395018400000000},
                       {0x000000006f76172e, 0x2bad0e9a00000000},
                       {0x000000003e2af045, 0xa125c29f00000000},
                       {0x000000005658d844, 0x80c9584f00000000},
                       {0x0000000064711657, 0xce6c873800000000},
                       {0x00000000827ae4f2, 0x0e8e8c8800000000},
                       {0x00000000ea938168, 0x335da4cd00000000},
                       {0x000000000dbc60c2, 0x2d9daa0f00000000},
                       {0x000000001e1255dd, 0x1e84ee2100000000}},
                      // Position 13
                      {{0x0000000000000000, 0x0000000000000000},
                       {0x0000000056f5cab9, 0xbfbcf49c00000000},
                       {0x00000000385993ac, 0xaec31a2f00000000},
                       {0x00000000a3d2cfe4, 0xf33519ee00000000},
                       {0x00000000b4247b20, 0x5c9cbdde00000000},
                       {0x00000000aee87806, 0xf5a4ad9800000000},
                       {0x00000000778a63ce, 0xe4ba555a00000000},
                       {0x00000000adcc94f4, 0x5df0e43d00000000},
                       {0x00000000695186a7, 0x9d57442800000000},
                       {0x0000000075cdb7ed, 0x0bdf6d5a00000000},
                       {0x00000000ce126d13, 0x6d4c124600000000},
                       {0x00000000c60d3fe6, 0xcb6f1f1a00000000},
                       {0x00000000bd97339c, 0x64eb408b00000000},
                       {0x00000000a65f7e5e, 0xaa4a448000000000},
                       {0x00000000a69617d3, 0x42ef789000000000},
                       {0x000000008d175638, 0x688d909000000000}},
                      // Position 14
                      {{0x0000000000000000, 0x0000000000000000},
                       {0x0000000066bc001e, 0x8a39e3c100000000},
                       {0x00000000b58b27b3, 0x85352d3c00000000},
                       {0x000000001acfdbfb, 0x7d0aaeb000000000},
                       {0x00000000fe273162, 0x291a605400000000},
                       {0x00000000a69a8eec, 0x027703b500000000},
                       {0x00000000f654a922, 0xcc6652b600000000},
                       {0x00000000b7620bb0, 0xe0dff1b100000000},
                       {0x00000000d3c98813, 0x53664be100000000},
                       {0x00000000da7a376f, 0x5b3d509700000000},
                       {0x00000000efa9f9ea, 0xe3879e8100000000},
                       {0x00000000a33b0ae3, 0x90a6d47c00000000},
                       {0x000000003d575eef, 0xf7a67e0100000000},
                       {0x0000000025f9e373, 0x428ed66200000000},
                       {0x000000001c9de7a4, 0x98f7ed2100000000},
                       {0x000000007660fe68, 0x67e5af6a00000000}},
                      // Position 15
                      {{0x0000000000000000, 0x0000000000000000},
                       {0x00000000f7d6deb4, 0x52b55e3900000000},
                       {0x00000000edb88320, 0x8000000000000000},
                       {0x000000007beb6f5a, 0xc4e22c3c00000000},
                       {0x0000000076dc4190, 0x4000000000000000},
                       {0x000000003df5b7ad, 0x6271161e00000000},
                       {0x000000003b6e20c8, 0x2000000000000000},
                       {0x00000000f34258f6, 0x31388b0f00000000},
                       {0x000000001db71064, 0x1000000000000000},
                       {0x0000000079a12c7b, 0xf524c6a700000000},
                       {0x000000000edb8832, 0x0800000000000000},
                       {0x00000000d168151d, 0x972ae07300000000},
                       {0x00000000076dc419, 0x0400000000000000},
                       {0x00000000850c89ae, 0xa62df31900000000},
                       {0x00000000ee0e612c, 0x0200000000000000},
                       {0x00000000428644d7, 0xbeae7aac00000000}}}};

checksums_constants aws_checksums_crc32c_constants = {
    .mu_poly = {0x0dea713f1, 0x0105ec76f1},
    .shift_factors = {// Position 0
                      {{0x0000000000000000, 0x0000000000000000},
                       {0x00000000f26b8303, 0x0100000000000000},
                       {0x0000000013a29877, 0x0001000000000000},
                       {0x00000000a541927e, 0x0000010000000000},
                       {0x00000000dd45aab8, 0x0000000100000000},
                       {0x0000000038116fac, 0xf26b830300000000},
                       {0x00000000ef306b19, 0x13a2987700000000},
                       {0x0000000068032cc8, 0xa541927e00000000},
                       {0x00000000493c7d27, 0xdd45aab800000000},
                       {0x00000000f43ed648, 0x38116fac00000000},
                       {0x00000000cb567ba5, 0xef306b1900000000},
                       {0x000000009771f7c1, 0x68032cc800000000},
                       {0x000000003171d430, 0x493c7d2700000000},
                       {0x0000000030d23865, 0xf43ed64800000000},
                       {0x0000000054075546, 0xcb567ba500000000},
                       {0x00000000678efd01, 0x9771f7c100000000}},
                      // Position 1
                      {{0x0000000000000000, 0x0000000000000000},
                       {0x00000000f20c0dfe, 0x3171d43000000000},
                       {0x000000003da6d0cb, 0xa2158b3400000000},
                       {0x000000001c291d04, 0x6051243f00000000},
                       {0x00000000740eef02, 0x75bba45b00000000},
                       {0x00000000083a6eec, 0x6d883e3800000000},
                       {0x00000000c49f4f67, 0x3365346a00000000},
                       {0x000000002ad91c30, 0x963e61cd00000000},
                       {0x000000006992cea2, 0x7417153f00000000},
                       {0x000000007e908048, 0xcf51951700000000},
                       {0x000000001b3d8f29, 0x3fc16b8600000000},
                       {0x00000000f1d0f55e, 0xc54608cd00000000},
                       {0x00000000a87ab8a8, 0x31c9460800000000},
                       {0x000000008462d800, 0x5706002200000000},
                       {0x0000000071d111a8, 0xad32746200000000},
                       {0x00000000ffd852c6, 0x048dc5cc00000000}},
                      // Position 2
                      {{0x0000000000000000, 0x0000000000000000},
                       {0x00000000dcb17aa4, 0x1426a81500000000},
                       {0x00000000bd6f81f8, 0xe986c14800000000},
                       {0x000000008d96551c, 0xc63764e600000000},
                       {0x00000000fe314258, 0xcdc220dd00000000},
                       {0x0000000074922601, 0xe312563600000000},
                       {0x000000001dfa0a15, 0x034a7d6300000000},
                       {0x00000000de8a97f8, 0x32d63d5c00000000},
                       {0x00000000f7506984, 0x1acaec5400000000},
                       {0x000000006e221adc, 0x5a7c508100000000},
                       {0x00000000b8b67c1c, 0x6ea6639f00000000},
                       {0x000000008ff2c75f, 0xe30bf1a900000000},
                       {0x00000000dada9a0b, 0xbec12e1e00000000},
                       {0x00000000b2b02c25, 0xad36163700000000},
                       {0x0000000068f95f1b, 0x74a4ebd400000000},
                       {0x00000000381978f1, 0x0466380200000000}},
                      // Position 3
                      {{0x0000000000000000, 0x0000000000000000},
                       {0x00000000c2a5b65e, 0x6bae74c400000000},
                       {0x00000000e040e0ac, 0x508c2ac800000000},
                       {0x00000000e1e5a161, 0x87ddf93e00000000},
                       {0x00000000c7cacead, 0x7b66d22300000000},
                       {0x000000000b89ad03, 0xbdb9bd7200000000},
                       {0x000000005d0c7db8, 0xca0b28a000000000},
                       {0x000000002ccae700, 0x9cde834800000000},
                       {0x0000000004fcdcbf, 0xc038134900000000},
                       {0x000000002e749a1c, 0xee9ec09500000000},
                       {0x00000000937bcf1c, 0xd2b4990e00000000},
                       {0x00000000675bfdd7, 0xe1e4d6a900000000},
                       {0x00000000dc6f9e57, 0x3172759d00000000},
                       {0x00000000c12ecc56, 0x2dd864dd00000000},
                       {0x00000000b46b1e6a, 0x8607972e00000000},
                       {0x000000008ce13f39, 0x987a7aca00000000}},
                      // Position 4
                      {{0x0000000000000000, 0x0000000000000000},
                       {0x000000006bafcc21, 0xf902f73d00000000},
                       {0x00000000140441c6, 0xf760b4e700000000},
                       {0x00000000d0b0f8bd, 0x44a6484f00000000},
                       {0x0000000068175a0a, 0x78409f1e00000000},
                       {0x00000000179343db, 0x81e514d200000000},
                       {0x000000000f4c6fb1, 0x6b5e4d6400000000},
                       {0x000000008b0aee8a, 0xb10df3bd00000000},
                       {0x00000000e1ff3667, 0xa70b0dc600000000},
                       {0x0000000093ea7d17, 0xf1db820600000000},
                       {0x00000000180ecbec, 0xe40b784d00000000},
                       {0x000000008f22d5bf, 0xda83e8cb00000000},
                       {0x000000008a2f08de, 0xd073116400000000},
                       {0x0000000037da0f70, 0xce5b12b300000000},
                       {0x00000000a5530278, 0x1f80c5ea00000000},
                       {0x000000009a97be0c, 0xaa8e81d300000000}},
                      // Position 5
                      {{0x0000000000000000, 0x0000000000000000},
                       {0x000000008b7230ec, 0xb2e4d22a00000000},
                       {0x0000000056175f20, 0xe69aa61200000000},
                       {0x00000000d4a30fa7, 0xb3da10db00000000},
                       {0x00000000b9a3dcd0, 0x7dd9f83700000000},
                       {0x00000000709b30fd, 0x23663f3a00000000},
                       {0x000000002f7f7bf2, 0xa82979c600000000},
                       {0x00000000bb6aad7e, 0xa9e5eef600000000},
                       {0x00000000dd2d789e, 0xfa0e459800000000},
                       {0x000000005a9e4e18, 0x70226ce100000000},
                       {0x000000005bef710b, 0x4715965600000000},
                       {0x000000004a87dc8c, 0xa1e2eaa700000000},
                       {0x00000000640eebda, 0xa8dbf4dc00000000},
                       {0x00000000780de59b, 0x7eed52d800000000},
                       {0x00000000ddc3e06f, 0x42b8047d00000000},
                       {0x0000000037513cfe, 0xe545e26100000000}},
                      // Position 6
                      {{0x0000000000000000, 0x0000000000000000},
                       {0x0000000044036c4a, 0xa0ffe38d00000000},
                       {0x000000004612657d, 0xd775258900000000},
                       {0x000000006246d48f, 0xd58aa71e00000000},
                       {0x00000000584d5569, 0x0b6e9e7e00000000},
                       {0x00000000e91135d8, 0x39fa33c100000000},
                       {0x00000000057d0a86, 0x29bc8f9a00000000},
                       {0x000000004b1b50ab, 0x753f60fa00000000},
                       {0x00000000e8cd33e2, 0x4fd0487500000000},
                       {0x00000000ea53f140, 0xf943740300000000},
                       {0x00000000afd9e2c5, 0x071aef1800000000},
                       {0x00000000243ed397, 0xbfc79e8800000000},
                       {0x000000001822aef3, 0x6a60ea7100000000},
                       {0x00000000c3fa4c11, 0x419771c600000000},
                       {0x00000000a9877b89, 0x9e09f00600000000},
                       {0x00000000de7070f8, 0xace8a6b000000000}},
                      // Position 7
                      {{0x0000000000000000, 0x0000000000000000},
                       {0x0000000082f63b78, 0x8000000000000000},
                       {0x00000000417b1dbc, 0x4000000000000000},
                       {0x0000000020bd8ede, 0x2000000000000000},
                       {0x00000000105ec76f, 0x1000000000000000},
                       {0x000000008ad958cf, 0x0800000000000000},
                       {0x00000000c79a971f, 0x0400000000000000},
                       {0x00000000e13b70f7, 0x0200000000000000},
                       {0x00000000f26b8303, 0x0100000000000000},
                       {0x00000000fbc3faf9, 0x0080000000000000},
                       {0x00000000ff17c604, 0x0040000000000000},
                       {0x000000007f8be302, 0x0020000000000000},
                       {0x000000003fc5f181, 0x0010000000000000},
                       {0x000000009d14c3b8, 0x0008000000000000},
                       {0x000000004e8a61dc, 0x0004000000000000},
                       {0x00000000274530ee, 0x0002000000000000}},
                      // Position 8
                      {{0x0000000000000000, 0x0000000000000000},
                       {0x0000000013a29877, 0x0001000000000000},
                       {0x00000000dd45aab8, 0x0000000100000000},
                       {0x00000000ef306b19, 0x13a2987700000000},
                       {0x00000000493c7d27, 0xdd45aab800000000},
                       {0x00000000cb567ba5, 0xef306b1900000000},
                       {0x000000003171d430, 0x493c7d2700000000},
                       {0x0000000054075546, 0xcb567ba500000000},
                       {0x00000000f20c0dfe, 0x3171d43000000000},
                       {0x000000000f69022b, 0x5407554600000000},
                       {0x000000003743f7bd, 0xf20c0dfe00000000},
                       {0x000000005c15eeb4, 0x0f69022b00000000},
                       {0x00000000ba4fc28e, 0x3743f7bd00000000},
                       {0x000000002dae840f, 0x5c15eeb400000000},
                       {0x00000000a2158b34, 0xba4fc28e00000000},
                       {0x0000000015bb4109, 0x2dae840f00000000}},
                      // Position 9
                      {{0x0000000000000000, 0x0000000000000000},
                       {0x000000003da6d0cb, 0xa2158b3400000000},
                       {0x00000000740eef02, 0x75bba45b00000000},
                       {0x00000000c49f4f67, 0x3365346a00000000},
                       {0x000000006992cea2, 0x7417153f00000000},
                       {0x000000001b3d8f29, 0x3fc16b8600000000},
                       {0x00000000a87ab8a8, 0x31c9460800000000},
                       {0x0000000071d111a8, 0xad32746200000000},
                       {0x00000000dcb17aa4, 0x1426a81500000000},
                       {0x000000006051d5a2, 0x25605e4000000000},
                       {0x0000000021f3d99c, 0x258d3fc900000000},
                       {0x00000000a00457f7, 0x87466f2100000000},
                       {0x0000000000ac29cf, 0xcfb6589400000000},
                       {0x0000000096638b34, 0xf4e995fd00000000},
                       {0x000000009af01f2d, 0xcb65cf9500000000},
                       {0x0000000088f25a3a, 0x8857e0fd00000000}},
                      // Position 10
                      {{0x0000000000000000, 0x0000000000000000},
                       {0x00000000bd6f81f8, 0xe986c14800000000},
                       {0x00000000fe314258, 0xcdc220dd00000000},
                       {0x000000001dfa0a15, 0x034a7d6300000000},
                       {0x00000000f7506984, 0x1acaec5400000000},
                       {0x00000000b8b67c1c, 0x6ea6639f00000000},
                       {0x00000000dada9a0b, 0xbec12e1e00000000},
                       {0x0000000068f95f1b, 0x74a4ebd400000000},
                       {0x00000000c2a5b65e, 0x6bae74c400000000},
                       {0x00000000deef32a2, 0xe9517b5d00000000},
                       {0x0000000044203757, 0x18398af900000000},
                       {0x00000000c52ab50d, 0x0c2b876100000000},
                       {0x00000000db808623, 0x9037dc1000000000},
                       {0x000000009714e4f9, 0xb3e90a5800000000},
                       {0x00000000abdbdde4, 0xda72e86300000000},
                       {0x00000000b4b2a045, 0xb1d4f7ff00000000}},
                      // Position 11
                      {{0x0000000000000000, 0x0000000000000000},
                       {0x00000000e040e0ac, 0x508c2ac800000000},
                       {0x00000000c7cacead, 0x7b66d22300000000},
                       {0x000000005d0c7db8, 0xca0b28a000000000},
                       {0x0000000004fcdcbf, 0xc038134900000000},
                       {0x00000000937bcf1c, 0xd2b4990e00000000},
                       {0x00000000dc6f9e57, 0x3172759d00000000},
                       {0x00000000b46b1e6a, 0x8607972e00000000},
                       {0x000000006bafcc21, 0xf902f73d00000000},
                       {0x0000000024ae8247, 0x537492a500000000},
                       {0x00000000a9454a97, 0x37415af500000000},
                       {0x000000003b902ae9, 0xa9584c0100000000},
                       {0x00000000abf8898d, 0xf16af94400000000},
                       {0x000000005b0489cf, 0x6151dbfe00000000},
                       {0x00000000f7354bff, 0x36d7edb300000000},
                       {0x00000000e1c81591, 0xb231f2a800000000}},
                      // Position 12
                      {{0x0000000000000000, 0x0000000000000000},
                       {0x00000000140441c6, 0xf760b4e700000000},
                       {0x0000000068175a0a, 0x78409f1e00000000},
                       {0x000000000f4c6fb1, 0x6b5e4d6400000000},
                       {0x00000000e1ff3667, 0xa70b0dc600000000},
                       {0x00000000180ecbec, 0xe40b784d00000000},
                       {0x000000008a2f08de, 0xd073116400000000},
                       {0x00000000a5530278, 0x1f80c5ea00000000},
                       {0x000000008b7230ec, 0xb2e4d22a00000000},
                       {0x0000000048fe2b39, 0xc8b6adeb00000000},
                       {0x00000000b6d8d6c2, 0x618d2cda00000000},
                       {0x00000000b12ec218, 0x82eaf23200000000},
                       {0x0000000024f8bb03, 0xb7d166dc00000000},
                       {0x0000000052ceee20, 0xb628de6c00000000},
                       {0x0000000047aaecfc, 0x858ac47800000000},
                       {0x00000000f7565d62, 0xa27c31f900000000}},
                      // Position 13
                      {{0x0000000000000000, 0x0000000000000000},
                       {0x0000000056175f20, 0xe69aa61200000000},
                       {0x00000000b9a3dcd0, 0x7dd9f83700000000},
                       {0x000000002f7f7bf2, 0xa82979c600000000},
                       {0x00000000dd2d789e, 0xfa0e459800000000},
                       {0x000000005bef710b, 0x4715965600000000},
                       {0x00000000640eebda, 0xa8dbf4dc00000000},
                       {0x00000000ddc3e06f, 0x42b8047d00000000},
                       {0x0000000044036c4a, 0xa0ffe38d00000000},
                       {0x000000006b0cb706, 0xf64a900000000000},
                       {0x00000000cf591942, 0x95ef60c900000000},
                       {0x0000000010bf976a, 0x759bb0a500000000},
                       {0x00000000d6031eeb, 0x4f15effc00000000},
                       {0x000000008c40bb2f, 0x00cb412b00000000},
                       {0x000000004f76d3c9, 0xfb7bbb4d00000000},
                       {0x00000000641f82df, 0xbf1d4f2900000000}},
                      // Position 14
                      {{0x0000000000000000, 0x0000000000000000},
                       {0x000000004612657d, 0xd775258900000000},
                       {0x00000000584d5569, 0x0b6e9e7e00000000},
                       {0x00000000057d0a86, 0x29bc8f9a00000000},
                       {0x00000000e8cd33e2, 0x4fd0487500000000},
                       {0x00000000afd9e2c5, 0x071aef1800000000},
                       {0x000000001822aef3, 0x6a60ea7100000000},
                       {0x00000000a9877b89, 0x9e09f00600000000},
                       {0x0000000082f63b78, 0x8000000000000000},
                       {0x00000000a1ff09c6, 0xe94ca9bc00000000},
                       {0x00000000aed091cc, 0x05b74f3f00000000},
                       {0x0000000002be8543, 0x14de47cd00000000},
                       {0x00000000746699f1, 0xa51e1f4200000000},
                       {0x00000000d51aca1a, 0x038d778c00000000},
                       {0x000000008ee76c01, 0xb7c64e4000000000},
                       {0x00000000d63586bc, 0x4f04f80300000000}},
                      // Position 15
                      {{0x0000000000000000, 0x0000000000000000},
                       {0x00000000417b1dbc, 0x4000000000000000},
                       {0x00000000105ec76f, 0x1000000000000000},
                       {0x00000000c79a971f, 0x0400000000000000},
                       {0x00000000f26b8303, 0x0100000000000000},
                       {0x00000000ff17c604, 0x0040000000000000},
                       {0x000000003fc5f181, 0x0010000000000000},
                       {0x000000004e8a61dc, 0x0004000000000000},
                       {0x0000000013a29877, 0x0001000000000000},
                       {0x00000000c76580d9, 0x0000400000000000},
                       {0x0000000070a27d8a, 0x0000100000000000},
                       {0x000000009edea41a, 0x0000040000000000},
                       {0x00000000a541927e, 0x0000010000000000},
                       {0x00000000aba65fe7, 0x0000004000000000},
                       {0x00000000e964b13d, 0x0000001000000000},
                       {0x000000007b2231f3, 0x0000000400000000}}}};

static uint32_t (*s_crc32c_fn_ptr)(const uint8_t *input, int length, uint32_t previous_crc32c) = NULL;
static uint32_t (*s_crc32_fn_ptr)(const uint8_t *input, int length, uint32_t previous_crc32) = NULL;

static uint32_t (*s_crc32_combine_fn_ptr)(uint32_t crc1, uint32_t crc2, uint64_t len) = NULL;
static uint32_t (*s_crc32c_combine_fn_ptr)(uint32_t crc1, uint32_t crc2, uint64_t len) = NULL;

void aws_checksums_crc32_init(void) {
    if (s_crc32_fn_ptr == NULL) {
#if defined(AWS_USE_CPU_EXTENSIONS) && defined(AWS_ARCH_ARM64)
        if (aws_cpu_has_feature(AWS_CPU_FEATURE_ARM_CRC)) {
            s_crc32_fn_ptr = aws_checksums_crc32_armv8;
        } else {
            s_crc32_fn_ptr = aws_checksums_crc32_sw;
        }
#else
        s_crc32_fn_ptr = aws_checksums_crc32_sw;
#endif
    }

    if (s_crc32c_fn_ptr == NULL) {
#if defined(AWS_USE_CPU_EXTENSIONS) && defined(AWS_ARCH_INTEL_X64)
        if (aws_cpu_has_feature(AWS_CPU_FEATURE_SSE_4_2)) {
            s_crc32c_fn_ptr = aws_checksums_crc32c_intel_avx512_with_sse_fallback;
        } else {
            s_crc32c_fn_ptr = aws_checksums_crc32c_sw;
        }
#elif defined(AWS_USE_CPU_EXTENSIONS) && defined(AWS_ARCH_ARM64)
        if (aws_cpu_has_feature(AWS_CPU_FEATURE_ARM_CRC)) {
            s_crc32c_fn_ptr = aws_checksums_crc32c_armv8;
        } else {
            s_crc32c_fn_ptr = aws_checksums_crc32c_sw;
        }
#else
        s_crc32c_fn_ptr = aws_checksums_crc32c_sw;
#endif
    }

    /* SW for now. still need to add hw versions. */
    if (s_crc32_combine_fn_ptr == NULL) {
        s_crc32_combine_fn_ptr = aws_checksums_crc32_combine_sw;
    }

    if (s_crc32c_combine_fn_ptr == NULL) {
        s_crc32c_combine_fn_ptr = aws_checksums_crc32c_combine_sw;
    }
}

uint32_t aws_checksums_crc32(const uint8_t *input, int length, uint32_t previous_crc32) {
    if (AWS_UNLIKELY(s_crc32_fn_ptr == NULL)) {
        aws_checksums_crc32_init();
    }
    return s_crc32_fn_ptr(input, length, previous_crc32);
}

uint32_t aws_checksums_crc32_ex(const uint8_t *input, size_t length, uint32_t previous_crc32) {
    return aws_large_buffer_apply_crc32(aws_checksums_crc32, input, length, previous_crc32);
}

uint32_t aws_checksums_crc32c(const uint8_t *input, int length, uint32_t previous_crc32c) {
    if (AWS_UNLIKELY(s_crc32c_fn_ptr == NULL)) {
        aws_checksums_crc32_init();
    }

    return s_crc32c_fn_ptr(input, length, previous_crc32c);
}

uint32_t aws_checksums_crc32c_ex(const uint8_t *input, size_t length, uint32_t previous_crc32) {
    return aws_large_buffer_apply_crc32(aws_checksums_crc32c, input, length, previous_crc32);
}

uint32_t aws_checksums_crc32_combine(uint32_t crc1, uint32_t crc2, uint64_t len2) {
    if (AWS_UNLIKELY(s_crc32_combine_fn_ptr == NULL)) {
        aws_checksums_crc32_init();
    }

    return s_crc32_combine_fn_ptr(crc1, crc2, len2);
}

uint32_t aws_checksums_crc32c_combine(uint32_t crc1, uint32_t crc2, uint64_t len2) {
    if (AWS_UNLIKELY(s_crc32c_combine_fn_ptr == NULL)) {
        aws_checksums_crc32_init();
    }

    return s_crc32c_combine_fn_ptr(crc1, crc2, len2);
}
